﻿// See https://aka.ms/new-console-template for more information

//#define VERBOSE

using System.Collections.Immutable;
using System.Reflection;
using System.Runtime.Loader;

class BepInExLoadContext : AssemblyLoadContext
{
    public static readonly IReadOnlyList<string> RelativeSearchPaths = ImmutableList.Create(
        "dotnet",
        Path.Combine("BepInEx", "core"),
        Path.Combine("BepInEx", "interop"),
        Path.Combine("BepInEx", "plugins", "DynamicLoader"));

    public static readonly string BepInExCoreName = "BepInEx.Core";
    public static readonly string BepInExPaths = "BepInEx.Paths";
    public delegate void Paths_SetExecutablePath(string executablePath, string? bepinRootPath = null, string? managedPath = null, bool gameDataRelativeToManaged = false, string[]? dllSearchPath = null);

    private readonly IReadOnlyList<string> searchPaths;

    public BepInExLoadContext(string exePath) : base(Path.GetFileName(exePath), isCollectible: true)
    {
        if (!File.Exists(exePath))
            throw new ArgumentException($"Could not find executable {exePath}", nameof(exePath));
        var gamePath = Path.GetDirectoryName(exePath);
        if (!Directory.Exists(gamePath))
            throw new ArgumentException($"Could not find game folder {gamePath}", nameof(exePath));
        Console.WriteLine(exePath);

        searchPaths = RelativeSearchPaths
            .Select(path => Path.Combine(gamePath, path))
            .ToImmutableList();
        if (!searchPaths.All(Directory.Exists))
            throw new ArgumentException($"Could not find BepInEx in path {gamePath}", nameof(exePath));

        var a_bepInExCore = Load(new AssemblyName(BepInExCoreName));
        if (a_bepInExCore == null)
            throw new ArgumentException($"Could not find {BepInExCoreName} in path {gamePath}", nameof(exePath));

        var t_bepInExPaths = a_bepInExCore.GetType(BepInExPaths);
        if (t_bepInExPaths == null)
            throw new ArgumentException($"Could not find {BepInExPaths} in {gamePath}", nameof(exePath));

        var m_setExecutablePath = t_bepInExPaths.GetMethod("SetExecutablePath", BindingFlags.Public | BindingFlags.Static);
        if (m_setExecutablePath == null)
            throw new ArgumentException($"Could not find {BepInExPaths}.SetExecutablePath() in {gamePath}", nameof(exePath));
        var setExecutablePath = m_setExecutablePath.CreateDelegate<Paths_SetExecutablePath>();
        setExecutablePath(exePath);
    }

    private static bool IsCompatible(AssemblyName targetName, AssemblyName potentialName)
    {
        var isSimpleName = targetName.Name == targetName.FullName;
        if (isSimpleName)
            return targetName.Name == potentialName.Name;

        var targetVersion = targetName.Version;
        var potentialVersion = potentialName.Version;

        targetName.Version = null;
        potentialName.Version = null;
        var isMatch = targetName.FullName == potentialName.FullName
            && potentialVersion >= targetVersion;

        targetName.Version = targetVersion;
        potentialName.Version = potentialVersion;
        return isMatch;
    }

    protected override Assembly? Load(AssemblyName assemblyName)
    {
        var loadedAssembly = AppDomain.CurrentDomain.GetAssemblies()
            .FirstOrDefault(ass => IsCompatible(assemblyName, ass.GetName()));
        if (loadedAssembly != null)
            return loadedAssembly;

        foreach (var searchPath in searchPaths)
        {
            var filePath = Path.Combine(searchPath, $"{assemblyName.Name}.dll");
            if (!File.Exists(filePath))
                continue;

            var name = AssemblyName.GetAssemblyName(filePath);
            if (!IsCompatible(assemblyName, name))
                continue;

#if VERBOSE
            Console.WriteLine($"Load {assemblyName.FullName}");
            Console.WriteLine($"\t{filePath}");
            Console.WriteLine($"\t\t{name.FullName}");
#endif
            return LoadFromAssemblyPath(filePath);
        }

        return null;
    }

    protected override IntPtr LoadUnmanagedDll(string unmanagedDllName)
    {
        foreach (var searchPath in searchPaths)
        {
            var filePath = Path.Combine(searchPath, unmanagedDllName);
            if (!File.Exists(filePath))
                continue;
#if VERBOSE
            Console.WriteLine($"LoadUnmanagedDll {unmanagedDllName}");
            Console.WriteLine($"\t{filePath}");
#endif
            return LoadUnmanagedDllFromPath(filePath);
        }

        return IntPtr.Zero;
    }
}
