﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.InteropServices;
using Basic.Reference.Assemblies;
using BepInEx;
using BepInEx.Logging;
using BepInEx.Unity.IL2CPP;
using HarmonyLib;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.Emit;
using Microsoft.CodeAnalysis.Text;
using Mono.Cecil;

namespace DynamicLoader;

[BepInPlugin(MyPluginInfo.PLUGIN_GUID, MyPluginInfo.PLUGIN_NAME, MyPluginInfo.PLUGIN_VERSION)]
public partial class Plugin : BasePlugin
{
    public static Plugin Instance { get; private set; }
    public static new ManualLogSource Log { get; private set; }
    private static Harmony patches = new Harmony(MyPluginInfo.PLUGIN_GUID);

    public override void Load()
    {
        Instance = this;
        Log = base.Log;

        patches.PatchAll();
        AddComponent<PluginComponent>();

        Log.LogInfo($"Plugin {MyPluginInfo.PLUGIN_NAME} is loaded!");
    }

    public void LoadIt()
    {
        var searchPaths = new List<string>(Directory.GetFiles(
            Paths.PluginPath, "*.cs", SearchOption.TopDirectoryOnly));
        searchPaths.AddRange(Directory.GetDirectories(Paths.PluginPath));
        searchPaths.AddRange(GetLinkedPaths());
        Log.LogInfo($"Searching\n\t{string.Join("\n\t", (IEnumerable<string>)searchPaths)}");

        foreach (var path in searchPaths)
            CompilePlugin(path);
    }

    public override bool Unload()
    {
        PluginLoadContext.UnloadAll();
        patches.UnpatchSelf();
        return base.Unload();
    }

    public static void CompilePlugin(string path)
    {
        Log.LogInfo(path);

        string[] csFiles;
        string assemblyName;
        if (File.Exists(path))
        {
            if (!path.EndsWith(".cs"))
                return;
            csFiles = new[] { path };
            assemblyName = Path.GetFileNameWithoutExtension(path);
        }
        else
        {
            csFiles = Directory.GetFiles(path, "*.cs", SearchOption.AllDirectories);
            if (csFiles.Length == 0)
                return;
            assemblyName = Path.GetFileName(path);
        }

        if (PluginLoadContext.Exists(assemblyName) || PluginUnloadContext.Exists(assemblyName))
        {
            Log.LogWarning($"{assemblyName} already exists");
            return;
        }

        var sourceTexts = csFiles.ToDictionary(csFile => csFile, csFile =>
            {
                using var csStream = File.OpenRead(csFile);
                return SourceText.From(csStream, canBeEmbedded: true);
            });
        var syntaxTrees = sourceTexts
            .Select(entry => CSharpSyntaxTree.ParseText(entry.Value, path: entry.Key))
            .ToList();
        var embeddedTexts = sourceTexts
            .Select(entry => EmbeddedText.FromSource(entry.Key, entry.Value))
            .ToList();

        using var assemblyResolver = new DefaultAssemblyResolver();
        assemblyResolver.AddSearchDirectory(Paths.BepInExAssemblyDirectory);
        /// <see cref="Il2CppInteropManager.IL2CPPInteropAssemblyPath" />
        assemblyResolver.AddSearchDirectory(Path.Combine(Paths.BepInExRootPath, "interop"));
        var frameworkRefs = Net60.References.All;
        var frameworkRefSet = frameworkRefs.Select(r => r.FilePath).ToHashSet();

        using var bepAss = AssemblyDefinition.ReadAssembly(typeof(BasePlugin).Assembly.Location);
        var bepAssRefs = bepAss.MainModule.AssemblyReferences.ToList();
        var bepRefDefs = bepAssRefs
            .Select(r =>
            {
                try
                {
                    return assemblyResolver.Resolve(r);
                }
                catch (AssemblyResolutionException ex)
                {
                    Log.LogWarning(ex.Message);
                    return null;
                }
            })
            .Where(d => d != null)
            .ToList();
        var bepRefFiles = bepRefDefs
            .Select(r => r.MainModule.FileName)
            .ToList();
        bepRefFiles.Add(bepAss.MainModule.FileName);
        var bepRefs = bepRefFiles
            .Where(f => !frameworkRefSet.Contains(Path.GetFileName(f)))
            .Select(f => MetadataReference.CreateFromFile(f))
            .ToList();
        foreach (var refDef in bepRefDefs)
            refDef.Dispose();

        Log.LogInfo("Creating compilation");

        var compilation = CSharpCompilation.Create(assemblyName)
            .WithOptions(options: new CSharpCompilationOptions(OutputKind.DynamicallyLinkedLibrary)
                .WithOptimizationLevel(OptimizationLevel.Debug))
            .AddSyntaxTrees(syntaxTrees)
            .WithReferences(frameworkRefs)
            .AddReferences(bepRefs);

        using MemoryStream assemblyStream = new(), symbolsStream = new();
        Log.LogInfo("Compiling");
        var emitResult = compilation.Emit(assemblyStream,
            pdbStream: symbolsStream,
            embeddedTexts: embeddedTexts,
            options: new EmitOptions(
                debugInformationFormat: DebugInformationFormat.PortablePdb));

        foreach (var diag in emitResult.Diagnostics)
        {
            if (!Enum.TryParse<LogLevel>(Enum.GetName(diag.Severity), true, out var level))
                level = LogLevel.Info;
            Log.Log(level, diag.ToString());
        }
        if (!emitResult.Success)
            return;

        assemblyStream.Seek(0, SeekOrigin.Begin);
        using var assemblyDefinition = AssemblyDefinition.ReadAssembly(assemblyStream,
            new ReaderParameters() { AssemblyResolver = assemblyResolver });

        Log.LogInfo("Finding PluginInfo");
        var pluginInfo = ToPluginInfo(assemblyDefinition);
        if (pluginInfo == null)
            return;

        if (PluginLoadContext.Exists(pluginInfo) || PluginUnloadContext.Exists(pluginInfo))
        {
            Log.LogWarning($"{pluginInfo.Metadata.GUID} already exists");
            return;
        }

        /*
        var pluginGuid = bepInPlugin.Metadata.GUID;
        var cacheDir = Path.Combine(Paths.CachePath, MyPluginInfo.PLUGIN_GUID);
        var outDir = Path.Combine(cacheDir, pluginGuid, "debug");
        if (Directory.Exists(outDir))
            Directory.Delete(outDir, true);
        Directory.CreateDirectory(outDir);

        var assemblyPath = Path.Combine(outDir, assemblyName + ".dll");
        var symbolsPath = Path.Combine(outDir, assemblyName + ".pdb");

        assemblyStream.Seek(0, SeekOrigin.Begin);
        using (var f = File.Open(assemblyPath, FileMode.Create))
            assemblyStream.CopyTo(f);

        symbolsStream.Seek(0, SeekOrigin.Begin);
        using (var f = File.Open(symbolsPath, FileMode.Create))
            symbolsStream.CopyTo(f);
        */

        var ctx = PluginLoadContext.Create(pluginInfo, assemblyName, assemblyStream, symbolsStream);
        if (!ctx.TryLoadPlugin())
        {
            Log.LogError($"Could not load {pluginInfo.Metadata.GUID}");
            ctx.Unload();
        }
    }

    public static PluginInfo ToPluginInfo(AssemblyDefinition assemblyDefinition)
    {
        foreach (var t in assemblyDefinition.MainModule.Types)
        {
            var info = IL2CPPChainloader.ToPluginInfo(t, null);
            if (info == null)
                continue;

            // Set location to an impossible value
            // We will intercept the FileNotFoundException
            Traverse.Create(info)
                .Property<string>(nameof(info.Location)).Value =
                    Path.Combine(typeof(Plugin).Assembly.Location, info.Metadata.GUID);
            return info;
        }
        return null;
    }

    public static IEnumerable<string> GetLinkedPaths()
    {
        if (!OperatingSystem.IsWindows())
            return Enumerable.Empty<string>();

        var links = Directory.GetFiles(Paths.PluginPath, "*.lnk", SearchOption.TopDirectoryOnly);
        if (links.Length == 0)
            return Enumerable.Empty<string>();

        var shell = COMObject.Create("Shell.Application");
        return links.Select(lnkFile =>
            shell.Invoke("NameSpace", Path.GetDirectoryName(lnkFile))
                .Invoke("Items")
                .Invoke("Item", Path.GetFileName(lnkFile))
                .Get("GetLink")
                .Get("Target")
                .Get<string>("Path"))
            .Where(Directory.Exists)
            .Distinct();
    }
}
