using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.CompilerServices;
using System.Runtime.InteropServices;
using System.Runtime.Loader;
using BepInEx;
using BepInEx.Unity.IL2CPP;
using HarmonyLib;
using UnityEngine.SceneManagement;

namespace DynamicLoader;

public class PluginUnloadContext
{
    private static readonly SortedDictionary<KeyType, PluginUnloadContext> UnloadingContexts = new (KeyType.COMPARER);
    private static readonly List<PluginUnloadContext> OrphanedContexts = new();
    private static readonly Stopwatch UnloadingStopwatch = new ();

    public static void Process()
    {
        if (UnloadingContexts.Count == 0)
            return;

        //Plugin.Log.LogInfo($"Processing {UnloadingContexts.Count} unloading and {OrphanedContexts.Count} orphaned contexts:\n\t{string.Join("\n\t", UnloadingContexts.Values.Select(ctx => ctx.ToString()))}");
        Plugin.Log.LogInfo($"Processing {UnloadingContexts.Count} unloading and {OrphanedContexts.Count} orphaned contexts");

        UnloadingStopwatch.Restart();
        Il2CppSystem.GC.Collect();
        Il2CppSystem.GC.InternalCollect(Il2CppSystem.GC.MaxGeneration);
        UnloadingStopwatch.Stop();
        Plugin.Log.LogInfo(FormattableString.Invariant($"Internal garbage collecting took {UnloadingStopwatch.Elapsed.TotalSeconds:0.###}s"));

        UnloadingStopwatch.Restart();
        GC.Collect();
        GC.WaitForPendingFinalizers();
        UnloadingStopwatch.Stop();
        Plugin.Log.LogInfo(FormattableString.Invariant($"Garbage collecting took {UnloadingStopwatch.Elapsed.TotalSeconds:0.###}s"));

        var unloaded = UnloadingContexts
            .Where(kv => kv.Value.IsUnloaded(incGC: true))
            .ToList()
            .Select(kv =>
            {
                UnloadingContexts.Remove(kv.Key);
                return kv.Value;
            })
            .Concat(OrphanedContexts
                .Where(ctx => ctx.IsUnloaded(incGC: true))
                .ToList()
                .Select(ctx =>
                {
                    OrphanedContexts.Remove(ctx);
                    return ctx;
                }))
            .ToList();

        if(unloaded.Count > 0)
            Plugin.Log.LogInfo($"Unloaded {unloaded.Count} contexts\n\t{string.Join("\n\t", unloaded.Select(ctx => ctx.ToString()))}");

        var orphaned = UnloadingContexts
            .Where(kv => kv.Value.GCs >= 20)
            .ToList()
            .Select(kv => {
                UnloadingContexts.Remove(kv.Key);
                return kv.Value;
            })
            .ToList();
        if (orphaned.Count > 0)
        {
            OrphanedContexts.AddRange(orphaned);
            Plugin.Log.LogWarning($"Orphaned {orphaned.Count} contexts\n\t{string.Join("\n\t", orphaned.Select(ctx => ctx.ToString()))}");
        }
    }

    public static bool Exists(PluginInfo info)
    {
        return UnloadingContexts.Values.Any(ctx => ctx.PluginInfo.Location == info.Location || ctx.PluginInfo.Metadata.GUID == info.Metadata.GUID);
    }

    public static bool Exists(string assName)
    {
        return UnloadingContexts.Values.Any(ctx => ctx.AssemblyName == assName);
    }

    public static void Create(PluginInfo info, string assName, WeakReference<AssemblyLoadContext> ctxRef, List<GCHandle> gcHandles)
    {
        var ctx = new PluginUnloadContext(info, assName, ctxRef, gcHandles);
        UnloadingContexts.Add(ctx.Key, ctx);
    }

    public PluginInfo PluginInfo { get; }
    public string AssemblyName { get; }
    public readonly DateTimeOffset UnloadStart = DateTimeOffset.UtcNow;
    public DateTimeOffset? UnloadDone { get; private set; } = null;
    public TimeSpan UnloadTime {
        get
        {
            if (UnloadDone.HasValue)
                return UnloadDone.Value - UnloadStart;
            return DateTimeOffset.UtcNow - UnloadStart;
        }
    }
    public bool Unloaded => UnloadDone != null;
    public int GCs { get; private set; }
    public readonly KeyType Key;

    private readonly WeakReference<AssemblyLoadContext> ctxRef;
    private readonly List<GCHandle> gcHandles;

    private PluginUnloadContext(PluginInfo info, string assName, WeakReference<AssemblyLoadContext> ctxRef, List<GCHandle> gcHandles)
    {
        PluginInfo = info;
        AssemblyName = assName;
        this.ctxRef = ctxRef;
        this.gcHandles = gcHandles;
        Key = new(PluginInfo.Metadata.GUID, UnloadStart);
    }

    public bool IsUnloaded(bool incGC = false)
    {
        if (!Unloaded)
        {
            if (!TryUseCtx())
                UnloadDone = DateTimeOffset.UtcNow;
            else if (incGC && GCs++ == 10)
                Uncuck();
        }
        return Unloaded;
    }

    public void Uncuck()
    {
        foreach (var gcHandle in gcHandles)
        {
            if (!gcHandle.IsAllocated)
                continue;
            Plugin.Log.LogWarning($"Freeing cucked handle {gcHandle.Target}");
            gcHandle.Free();
        }
        gcHandles.Clear();
    }

    public List<string> GetLoadedAssemblies()
    {
        var ret = new List<string>();
        TryUseCtx(ctx => ret
            .AddRange(ctx.Assemblies
                .Select(ass => ass.FullName)
                .OrderBy(x => x, StringComparer.OrdinalIgnoreCase)));
        return ret;
    }

    [MethodImpl(MethodImplOptions.NoInlining)]
    private bool TryUseCtx(Action<AssemblyLoadContext> action = null)
    {
        if (!ctxRef.TryGetTarget(out var ctx))
            return false;
        action?.Invoke(ctx);
        return true;
    }

    public override string ToString()
    {
        string asses = "";
        if (!Unloaded)
        {
            var names = GetLoadedAssemblies();
            asses = $"\n\t\tLoaded assemblies: {names.Count}";
            if (names.Count > 0)
                asses = $"{asses}\n\t\t\t{string.Join("\n\t\t\t", names)}";
        }
        return FormattableString.Invariant($"{PluginInfo.Metadata.GUID}:\n\t\tName: {PluginInfo.Metadata.Name}\n\t\tGCs survived: {GCs}\n\t\tUnloadTime: {UnloadTime.TotalSeconds:0.###}s{asses}");
    }

    public readonly struct KeyType
    {
        public static readonly IComparer<KeyType> COMPARER =
            Comparer<KeyType>.Create((left, right) => {
                var cmp = StringComparer.OrdinalIgnoreCase.Compare(left.GUID, right.GUID);
                if (cmp != 0)
                    return cmp;
                return left.UnloadStart.CompareTo(right.UnloadStart);
            });

        public readonly string GUID;
        public readonly DateTimeOffset UnloadStart;

        public KeyType(string guid, DateTimeOffset unloadStart)
        {
            GUID = guid;
            UnloadStart = unloadStart;
        }
    }
}

public class PluginLoadContext : IDisposable
{
    public enum LoadStates
    {
        READY,
        LOADING,
        LOADED,
        DISPOSED,
    }

    public static Dictionary<string, PluginLoadContext> ContextsByLocation { get; } = new();

    public static Assembly LoadPlugin(string assemblyFile)
    {
        if (ContextsByLocation.TryGetValue(assemblyFile, out var ctx))
            return ctx.LoadOrGetPluginAssembly(assemblyFile);
        return null;
    }

    public static void UnloadAll()
    {
        foreach (var ctx in ContextsByLocation.Values)
            ctx.Unload();
        ContextsByLocation.Clear();
    }

    public static bool Exists(PluginInfo info)
    {
        if (ContextsByLocation.ContainsKey(info.Location))
            return true;
        return ContextsByLocation.Values.Any(ctx => ctx.PluginInfo.Metadata.GUID == info.Metadata.GUID);
    }

    public static bool Exists(string assName)
    {
        return ContextsByLocation.Values.Any(ctx => ctx.AssemblyName == assName);
    }

    public static PluginLoadContext Create(PluginInfo info, string assName, MemoryStream assemblyStream, MemoryStream symbolsStream)
    {
        if (Exists(info))
            throw new InvalidOperationException($"{info.Metadata.GUID} already exists in dict");
        var ctx = new PluginLoadContext(info, assName, assemblyStream, symbolsStream);
        ContextsByLocation[info.Location] = ctx;
        return ctx;
    }

    private delegate IList<PluginInfo> IL2CPPChainLoader_LoadPlugins(IList<PluginInfo> plugins);
    private static readonly IL2CPPChainLoader_LoadPlugins LoadPlugins;

    static PluginLoadContext()
    {
        var m_LoadPlugins = typeof(IL2CPPChainloader).BaseType
            .GetMethod("LoadPlugins", BindingFlags.NonPublic | BindingFlags.Instance, new[] { typeof(IList<PluginInfo>) });
        LoadPlugins = m_LoadPlugins.CreateDelegate<IL2CPPChainLoader_LoadPlugins>(IL2CPPChainloader.Instance);
    }

    public PluginInfo PluginInfo { get; private set; }
    public string AssemblyName { get; }
    public LoadStates LoadState { get; private set; } = LoadStates.READY;
    public string Name { get; }

    private MemoryStream assemblyStream, symbolsStream;
    private AssemblyLoadContext ctx;
    private Assembly assembly;
    private List<GCHandle> gcHandles;

    private PluginLoadContext(PluginInfo info, string assName, MemoryStream assemblyStream, MemoryStream symbolsStream)
    {
        PluginInfo = info;
        AssemblyName = assName;
        this.assemblyStream = assemblyStream;
        this.symbolsStream = symbolsStream;

        ctx = new AssemblyLoadContext(info.Metadata.GUID, isCollectible: true);
        Name = ctx.Name;
        ctx.Unloading += OnUnloading;
    }

    public bool TryLoadPlugin()
    {
        if (LoadState != LoadStates.READY)
            throw new InvalidOperationException($"Can not load plugin {PluginInfo.Metadata.GUID} "
                + $"in state {Enum.GetName(LoadState)}, expected {Enum.GetName(LoadStates.READY)}");
        if (PluginInfo.Instance != null)
            throw new InvalidOperationException($"Can not load plugin {PluginInfo.Metadata.GUID} "
                + $"because it already has an instance associated with it ({PluginInfo.Instance.GetType().FullName})");

        Plugin.Log.LogInfo($"Loading plugin {PluginInfo.Metadata.Name} [{PluginInfo.Metadata.GUID}]");
        LoadState = LoadStates.LOADING;
        Patches.fuckOff.Clear();
        Patches.Collect = true;
        var loadedPlugins = LoadPlugins(new List<PluginInfo>() { PluginInfo });
        Patches.Collect = false;
        gcHandles = Patches.fuckOff.ToList();

        if (LoadState != LoadStates.LOADED)
        {
            var state = LoadState;
            if (state != LoadStates.DISPOSED)
                ctx?.Unload();
            if (state == LoadStates.LOADING)
                throw new InvalidOperationException($"{nameof(IL2CPPChainloader)} did not load plugin {PluginInfo.Metadata.GUID}, "
                    + "is Harmony patch working?");
            else
                throw new InvalidOperationException($"Unexpected load state {Enum.GetName(state)} after loading "
                + $"plugin {PluginInfo.Metadata.GUID}, expected {Enum.GetName(LoadStates.LOADED)}");
        }

        return PluginInfo.Instance is BasePlugin;
    }

    public Assembly LoadOrGetPluginAssembly(string assemblyFile)
    {
        if (PluginInfo.Location != assemblyFile)
            return null;

        if (LoadState == LoadStates.LOADED && assembly != null)
            return assembly;

        if (LoadState != LoadStates.LOADING)
            return null;

        assemblyStream.Seek(0, SeekOrigin.Begin);
        symbolsStream.Seek(0, SeekOrigin.Begin);
        assembly = ctx.LoadFromStream(assemblyStream, symbolsStream);
        assemblyStream = null;
        symbolsStream = null;
        LoadState = LoadStates.LOADED;
        Plugin.Log.LogInfo($"Assembly {assembly.GetName().Name} [{PluginInfo.Metadata.GUID}] loaded from in-memory stream");
        return assembly;
    }

    public class Cumger
    {
        public static Cumger<T, object> Create<T>(T obj) where T : class
        {
            return new Cumger<T, object>(obj);
        }
    }

    public class Cumger<T, TData> where T : class
    {
        public static readonly Cumger<T, TData> Empty = new(null) { suppress = true };
        
        public T Value { get; }
        public TData Data { get; }
        private bool suppress;
        public Cumger(T obj)
        {
            Value = obj;
        }
        public Cumger((T, TData) obj)
        {
            Value = obj.Item1;
            Data = obj.Item2;
        }
        
        public Cumger<T, TData> WithSuccess(Action success)
        {
            return With((value, data) => success(), false);
        }
        public Cumger<T, TData> WithSuccess(Action<T> success)
        {
            return With((value, data) => success(value), false);
        }
        public Cumger<T, TData> WithSuccess(Action<T, TData> success)
        {
            return With(success, false);
        }
        public Cumger<T, TData> WithError(Action error)
        {
            return With((value, data) => error(), true);
        }
        public Cumger<T, TData> WithError(Action<TData> error)
        {
            return With((value, data) => error(data), true);
        }

        private Cumger<T, TData> With(Action<T, TData> action, bool eqNull)
        {
            if (!suppress && (Value == null) == eqNull)
                action(Value, Data);
            return this;
        }

        public Cumger<TRet, TData> Then<TRet>(Func<T, TRet> f) where TRet : class
        {
            if (Value == null) return Cumger<TRet, TData>.Empty;
            return new Cumger<TRet, TData>(f(Value));
        }
        public Cumger<TRet, TRetData> Then<TRet, TRetData>(Func<T, (TRet, TRetData)> f) where TRet : class
        {
            if (Value == null) return Cumger<TRet, TRetData>.Empty;
            return new Cumger<TRet, TRetData>(f(Value));
        }
        public Cumger<TRet, TData> Then<TRet>(Func<T, TData, TRet> f) where TRet : class
        {
            if (Value == null) return Cumger<TRet, TData>.Empty;
            return new Cumger<TRet, TData>(f(Value, Data));
        }
        public Cumger<TRet, TRetData> Then<TRet, TRetData>(Func<T, TData, (TRet, TRetData)> f) where TRet : class
        {
            if (Value == null) return Cumger<TRet, TRetData>.Empty;
            return new Cumger<TRet, TRetData>(f(Value, Data));
        }
    }

    public void Unload()
    {
        ctx?.Unload();
    }

    private void OnUnloading(AssemblyLoadContext unloadingCtx)
    {
        if (unloadingCtx != ctx)
            return;
        ctx.Unloading -= OnUnloading;

        Plugin.Log.LogInfo("OnUnloading " + ctx.Name);
        var oldState = LoadState;
        LoadState = LoadStates.DISPOSED;
        assemblyStream = symbolsStream = null;
        ContextsByLocation.Remove(PluginInfo.Location);

        gcHandles = gcHandles
            .Where(gcHandle =>
            {
                if (!gcHandle.IsAllocated)
                    return false;
                if (gcHandle.Target.ToString() == "Il2CppInterop.Runtime.Injection.ClassInjector+InvokerDelegate")
                    return true;
                Plugin.Log.LogWarning($"Freeing cucked handle {gcHandle.Target}");
                gcHandle.Free();
                return false;
            })
            .ToList();
        PluginUnloadContext.Create(PluginInfo, AssemblyName, new WeakReference<AssemblyLoadContext>(ctx), gcHandles);
        gcHandles = null;

        if (oldState == LoadStates.LOADED)
        {
            // "Unload" references in Unity Explorer
            var t_consoleController = Cumger.Create(IL2CPPChainloader.Instance.Plugins.GetValueOrDefault("com.sinai.unityexplorer")?.Instance)
                    .WithError(() => Plugin.Log.LogWarning("UnityExplorer is not loaded"))
                .Then(exp => exp.GetType().Assembly.GetType("UnityExplorer.CSConsole.ConsoleController"))
                    .WithError(() => Plugin.Log.LogWarning("Could not find type [UnityExplorer.CSConsole.ConsoleController]"))
                .Value;
            if (t_consoleController != null)
            {
                Plugin.Log.LogInfo("[UnityExplorer.CSConsole.ConsoleController] found, clearing references");

                var stdLib = Cumger.Create(t_consoleController)
                    .Then(cc => cc.GetProperty("Evaluator", BindingFlags.Public | BindingFlags.Static)?.GetValue(null))
                        .WithError(() => Plugin.Log.LogWarning("Could not find property [public static ConsoleController.Evaluator]"))
                    .Then(eval => eval.GetType().GetField("StdLib", BindingFlags.NonPublic | BindingFlags.Static)?.GetValue(null) as HashSet<string>)
                        .WithError(() => Plugin.Log.LogWarning("Could not find field [private static Evaluator.StdLib]"))
                    .Value;
                if (stdLib != null && stdLib.Add(AssemblyName))
                {
                    Plugin.Log.LogInfo($"{AssemblyName} added to UnityExplorer ignore list");

                    Cumger.Create(t_consoleController.GetMethod("ResetConsole", BindingFlags.Public | BindingFlags.Static, Type.EmptyTypes))
                            .WithError(() => Plugin.Log.LogWarning("Could not find method [public static ConsoleController.ResetConsole()]"))
                        .Then(m_reset => m_reset.Invoke(null, null));

                    if (stdLib.Remove(AssemblyName))
                        Plugin.Log.LogInfo($"{AssemblyName} removed from UnityExplorer ignore list");
                }
            }

            // "Unload" references in UniverseLib
            var allTypes = Cumger.Create(AppDomain.CurrentDomain.GetAssemblies()
                .FirstOrDefault(ass => ass.GetName().Name == "UniverseLib.IL2CPP.Interop"))
                    .WithError(() => Plugin.Log.LogWarning("UniverseLib not loaded"))
                .Then(uLib => uLib.GetType("UniverseLib.ReflectionUtility"))
                    .WithError(() => Plugin.Log.LogWarning("Could not find type [UniverseLib.ReflectionUtility]"))
                .Then(refUtil => refUtil.GetField("AllTypes", BindingFlags.Public | BindingFlags.Static)?.GetValue(null) as IDictionary<string, Type>)
                    .WithError(() => Plugin.Log.LogWarning("Could not find field [public static UniverseLib.ReflectionUtility.AllTypes]"))
                .Value;
            if (allTypes != null)
            {
                var ourTypes = allTypes
                    .Where(t => ctx.Assemblies.Contains(t.Value.Assembly))
                    .ToList();
                Plugin.Log.LogInfo($"AllTypes contains {ourTypes.Count} of our types, removing\n{string.Join("\n", ourTypes?.Select(t => t.Value.FullName))}");
                foreach (var kv in ourTypes)
                    allTypes.Remove(kv.Key);
            }
        }

        // Release references to anything in the context
        assembly = null;
        ctx = null;

        if (PluginInfo.Instance is BasePlugin plugin)
        {
            try
            {
                plugin.Unload();
            }
            catch (Exception ex)
            {
                Plugin.Log.LogWarning($"Exception thrown while unloading plugin {PluginInfo.Metadata.Name} [{PluginInfo.Metadata.GUID}], "
                    + $"{nameof(AssemblyLoadContext)} might not unload properly\n{ex.ToString()}");
            }
        }

        Traverse.Create(PluginInfo).Property<object>(nameof(PluginInfo.Instance)).Value = null;
        IL2CPPChainloader.Instance.Plugins.Remove(PluginInfo.Metadata.GUID);
    }

    public void Dispose()
    {
        Unload();
        GC.SuppressFinalize(this);
    }

    [HarmonyPatch]
    public static class Patches
    {
        [HarmonyFinalizer]
        [HarmonyPatch(typeof(Assembly), nameof(Assembly.LoadFrom), typeof(string))]
        static Exception Assembly_LoadFrom(string assemblyFile, ref Assembly __result, Exception __exception)
        {
            if (__exception is not FileNotFoundException)
                return __exception;
            try
            {
                __result = LoadPlugin(assemblyFile);
                return __result == null ? __exception : null;
            }
            catch (Exception ex)
            {
                return new AggregateException("Could not intercept Assembly.LoadFrom", new List<Exception>() { __exception, ex });
            }
        }

        public static bool Collect = false;
        public static List<GCHandle> fuckOff = new();

        [HarmonyPostfix]
        [HarmonyPatch(typeof(GCHandle), nameof(GCHandle.Alloc), typeof(object))]
        static void GCHandle_Alloc(object value, ref GCHandle __result)
        {
            if (!Collect)
                return;
            Plugin.Log.LogWarning($"GCHandle.Alloc({value?.GetType()?.FullDescription() ?? "null"}) => {__result.ToString()}");
            if (value is Delegate dlgt)
                Plugin.Log.LogError($"Delegate {dlgt.Method?.Name}");
            fuckOff.Add(__result);
        }

        [HarmonyPostfix]
        [HarmonyPatch(typeof(GCHandle), nameof(GCHandle.Alloc), typeof(object), typeof(GCHandleType))]
        static void GCHandle_Alloc(object value, GCHandleType type, ref GCHandle __result)
        {
            if (!Collect)
                return;
            Plugin.Log.LogWarning($"GCHandle.Alloc({value?.GetType()?.FullDescription() ?? "null"}, {Enum.GetName(type)}) => {__result.ToString()}");
            if (value is Delegate dlgt)
                Plugin.Log.LogError($"Delegate {dlgt.Method?.Name}");
            fuckOff.Add(__result);
        }
    }
}
